/**
 *
 * \file        cresnet_slave.c
 *
 * \brief       Cresnet stuff that is used by BOTH "uart" cresnet and
 *              Cresnet over IP (CIP).  Implements the "slave" end of Cresnet,
 *              not the "master" end.
 *
 * \author      Pete McCormick (copied from network.c in Cajita code)
 *
 * \date        12/5/2007
 *
 * \note        This should be a generic module.  We support 2 cresnet ports,
 *              but 1 has to be slave and 1 has to be master.  You can't
 *              have 2 slaves or 2 masters at the same time.  Sorry.
 */

////////////////////////////////////////////////////////////////////////////////

#include "Cresnet.h"
#include "cresnet_slave.h"
#include "errors.h"
#include <string.h>
#include "os.h"
#include "uart_cresnet_slave.hpp"
#include "charconv.h"
#include "console.h"
#include "tji.h"
#include <ctype.h>
#include "cip.h"
#include "memorymanager.h"
#include <stdlib.h>
#include "product.h"
#include "field_debug.h"
#include "cresnet_host.hpp"
#include "DMCardJoins.h"
#include "serialport.h"
#include "cnet_pnp.h"
#include "dm_netmapper.hpp"
#include "Cresnetconsole.hpp"
#include "StreamMapper.h"
#include "hardware.h"
#include "CresnetRconTimer.h"

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT

#include "CDMSwitchCard.h"
#include "DMController.h"
#include "CDmNetDiscover.h"

#endif

#ifdef STR91X_IAR
#include <intrinsics.h>
#endif // STR91X_IAR

// custom offset is defined and (-1) such that it gets packed in a
// bit field as 11 binary when it is extracted from the bit field
// into an unsigned it becomes 3 so comparisions to NET_DEST_CUSTOM
// need to add 4 to the minus 1 for the comparison to work
#define CUSTOM_OFFSET 4

// Pointer to functions for PnP capabilities
// If PnP is supported, these are initialized to the PNP routines
// otherwise they are 0 and the commands are not processed.
PCNETLIGHT pfClearLightAndPoll;
PCNETCHECKTSID pfCheckTouchSettableIDPacket;
PCPNPPARSE pfPNPRxParse;
CNET_ENDOFUPDATE pFnEndOfUpdate = NULL;
PCNET_PACKET2C pfProcessPacketType_2C = NULL;

// Function pointer to intercept packets before being parsed by CresnetSlaveRxParse()
// Intercepted packets will not be forwarded to endpoints
extern BOOL (*pfPreCnetSlaveParsePktIntercept)(UINT8* pPkt, UINT8 PktLen, UINT8 PktType, UINT8 Source) = NULL;

// Function pointer to process packets after being parsed by CresnetSlaveRxParse()
// makes it easier to handle new packet types without affecting the main parser
extern void (*pfPostCnetSlaveParsePktProcess)(UINT8* pPkt, UINT8 PktLen, UINT8 PktType, UINT8 Source) = NULL;
////////////////////////////////////////////////////////////////////////////////

static CRESNETSLAVE CresnetSlave;
UINT32 CresnetSlaveLockId;
UINT8 NextType;
CresnetJoinState **pJoinState;
CCresnetConsole *CresnetConsole;
// pointer to a function for a custom cresnet destination
CNET_CUSTOM m_pfCnetCustom;


#ifdef STR91X_IAR
bool gbBufferingVectorTableSRecords = false;
// pointer to the linked list of bootloader S records,
// used only during bootloader upgrade
static UINT8 *gpBootloaderVectorArea[2] = {NULL, NULL};
static UINT32 gBootloaderVectorAreaByteCnt[2];

// this is only used during bootloader upgrade, initialized by first call to WriteBootloaderFlashRecord
static CNETHELPER *gpCnetHelper = NULL;

extern BOOL gbWritingBootloaderVectorArea;

void WriteBootloaderVectorBlockIfNeeded(void);  // forward declaration
#endif // STR91X_IAR
void (*SetTimeforRTC)(short hour, short minute, short second ) = NULL;
void (*SetDateforRTC)(short month, short day,   short year   ) = NULL;

// Function pointer: What handles each Data Transfer "Block Complete" message.
static CNET_XFER_BLOCK_COMPLETE_FTYPE pfCresnetDataXferBlockComplete = NULL;
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
extern BOOL DMMidpointControllerHandleDMDeviceConfigurationPacket(CREST_ALL_PACKETS *pCresnetPacket);
#endif

////////////////////////////////////////////////////////////////////////////////

#ifdef DM_V24
 void CresnetSlaveProcessSleep();  //Project specific functions,turn off LCD
 void CresnetSlaveProcessWakeup(); //Project specific functions,turn on LCD
#endif

void CresnetSlaveResetTx(UINT32 source)
{
    // blah blah
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       process CNET_COMMAND packets as a slave
 * \return      none
 * \retval      void
 * \param       packet - pointer to the cresnet packet to be parsed
 * \param       source - where the packet came from
 * \param       maxlen - maximum size the packet can be
 * \param       slot - slot number
*/
void CresnetSlaveRxParseCmd(UINT8 *packet, UINT32 source, UINT32 maxlen, UINT32 slot)
{
    switch(*(packet+3))
    {
        case CNET_CMD_ALL_CLEAR:
        case CNET_CMD_ALL_CLEAR_PROG_SPECIFIC:
            //DmSystemError(DM_ERROR_LEVEL_NOTICE,DM_ERROR_SUBSYS_CNET,ERR_CNET_REC_ALL_CLEAR,0);
            CresnetSlave.allClearCnt++;
            CresnetSlaveResetTx(source);
            //  On broadcast all clear, the HPRFGW issues update request, the control system will then send
            //  the device specific all clear followed by extender status request

           // Fix bug 37724: All Clear BROADCAST doesn't stop the process.
            if (pfClearLightAndPoll)
            {
              (*pfClearLightAndPoll)();
            }

            //  The broadcast all clear command is issued when the control system reboots either due to
            //  loading a new Simple program etc...
            if(packet[CNET_ID_OFFSET] == BROADCAST)
            {
                // don't expect to get this over CIP
                if (GetCresnetSlaveUart())
                   GetCresnetSlaveUart()->UartCresnetSlaveSetRestartFlag(1);
            }
            else
            {
                // if its a program specific all clear,
                if (*(packet+3) == CNET_CMD_ALL_CLEAR_PROG_SPECIFIC)
                    // pass the program number
                    CresnetSlaveProcessAllClear(*(packet+4), (UINT8)source);
                else        // since this is a broadcast, we will only see it once, clear all streams
                    CresnetSlaveProcessAllClear(ALL_STREAMS, (UINT8)source);
            }
            break;

        case CNET_CMD_END_OF_UPDATE:
            //DmSystemError(DM_ERROR_LEVEL_NOTICE,DM_ERROR_SUBSYS_CNET,ERR_CNET_REC_END_UPDATE,0);
            if (pFnEndOfUpdate)
                pFnEndOfUpdate (ALL_STREAMS, (UINT8)source);
        break;
        case CNET_CMD_END_OF_UPDATE_PROG_SPECIFIC:
            // end of update message for a specific program (for shareable devices)
            if (pFnEndOfUpdate)
                pFnEndOfUpdate (*(packet+4), (UINT8)source);
        break;

        case CNET_CMD_SELECTIVE_ALLCLEAR:
             if (pfClearLightAndPoll)
                  (*pfClearLightAndPoll)();
            break;

        case CNET_CMD_NAME_REQ:
            // only expect this for uart cresnet
            if (GetCresnetSlaveUart())
            {
              GetCresnetSlaveUart()->UartCresnetSlaveSetAutoId(GetStreamFromPacket(*packet));
            }
            //Bugzilla fix
            //Add support for name request response on ARM USB IOP projects
            else if ( pDMNetMapper )
            {
                UINT8 length = 0;
                UINT8* pBuffer = NULL;
                if ( MemMgr )
                {
                    pBuffer = MemMgr->GetBlock(CNET_MAX_MSG_SIZE);
                }
                if ( pBuffer )
                {
                    FirmwareGetName((char *)&pBuffer[5], CNET_MAX_MSG_SIZE-6, 0);                    

                    length = strlen((char *)&pBuffer[5]) +  3;

                    /* now fill in the beginning of the packet */
                    pBuffer[0]= CNET_ID_CTRLSYS;
                    pBuffer[1]= length;
                    pBuffer[2]= NAME_REQ;
                    pBuffer[3]= 0;
                    pBuffer[4]= 0;

                    pDMNetMapper->QueuePacketToController(GetStreamFromPacket(*packet), (void*)pBuffer, pBuffer[1] + 2);

                    MemMgr->FreeBlock(pBuffer);
                }
            }
            break;

        case CNET_CMD_REQUEST_END:
        {
            //Start the command request ACK timer if possible
            if ( GetState(GetStreamFromPacket(*packet)) )
            {
                GetState(GetStreamFromPacket(*packet))->StartCommandRequestAckTimer(source);
            }
            break;
        }
#ifdef DM_V24
        case CNET_CMD_SLEEP:
        {
            CresnetSlaveProcessSleep();
            break;
        }
        case CNET_CMD_WAKE:
        {
            CresnetSlaveProcessWakeup();
            break;
        }
#endif

        default:  /* unknown packet type - do nothing */
            //DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_REC_UNKNOWN_CMD,packet[3]);
            break;
    }
}

// stick something in uninitialized RAM? or somewhere else to indicate to
// bootloader what to do.
void CresnetBootloaderRequest(BOOL doBootloader, UINT8 bStream)
{
    UINT8 *pCmdBlock = GetBootCmdPtr();
#ifdef APPLICATION_CLEANUP
    //  Note: to enable the application cleanup, set the APPLICATION_CLEANUP parameter in the Crestore IAR project file and define this function in the
    //  application. It was done this way not as a call or a pointer due to RAM and ROM space limitations in some DM cards.
    //  This function would prepare application cleanup such as memory release, stop processing or save eeprom parameters etc...
    ApplicationCleanup();
#endif
    // make sure any non-volatile data is written to memory
    UpdateSystemNVL();

    // turn off interrupts;  we will reboot after this.
    OsEnterCritical();

    // write the upgrade command at the beginning of RAM
    strcpy((char *)pCmdBlock, UPGRADE_FIRMWARE);
    // store the cresnet address after the command
    pCmdBlock += UPGRADE_CMD_LEN;
    *(UINT8*)pCmdBlock = CresnetGetDefaultId() + bStream;

    // reboot to come up in bootloader
    DmRebootNotFatal();
}

void CresnetSlaveSendReadyForXfer(UINT32 blockBytes, UINT8 bStream)
{
    UINT8 packet[5];
    UINT8 packetBytes;

    if(blockBytes == 0)
    {
        // not specifying block size
        packetBytes = 4;
    }
    else
    {
        packetBytes = 5;
        // copied from Jiwan
        // block bytes as 2^n
        packet[4] = 0;
        blockBytes --;
        while(blockBytes)
        {
            blockBytes >>= 1;
            packet[4]++;
        }
    }
    packet[0] = CNET_ID_CTRLSYS;
    packet[1] = packetBytes-2;  // len
    packet[2] = CNET_GLOBAL;
    packet[3] = DNLD_CTRL_PREPARE_SCREEN;
    // since its global, use the first stream
    CresnetSlaveEnqueueOutgoingPacket(bStream, packet, packetBytes, NET_DEST_CRESNET);
}

/**
* \brief    Sets the function pointer for what to call upon receiving each
*           Data Transfer "Block Complete" message, allowing an application
*           to define its own handler for that kind of message.
* \param    funcPtr - Points to what function to call. NULL resets to default.
* \return   (None).
* \author   Arto Kiremitdjian.
* \date     6/29/2012
*/
void SetFuncForCnetXferBlockComplete (CNET_XFER_BLOCK_COMPLETE_FTYPE funcPtr)
{
    pfCresnetDataXferBlockComplete = funcPtr;
}

/**
 * \author      Pete McCormick
 * \date        11/1/2005
 * \return      void
 *
 * \brief       parse type 4 packets
 *
 * \param       packet
 * \param       maxlen
 * \param       slot
 * \param       source
 */
void CresnetSlaveRxParseGlobal(UINT8 *packet, UINT32 source, UINT32 maxlen, UINT32 slot)
{
    switch(*(packet+3))
    {
        case DNLD_CTRL_READY:
            CresnetSlaveSendReadyForXfer(CresnetSlave.blockBytes, GetStreamFromPacket(packet[0]));
            break;

        case DNLD_CTRL_DONE:
            CresnetDataXferEnd(GetStreamFromPacket(packet[0]));
            EnableNVLWrites();
            break;

        case DNLD_CTRL_SCREEN_DUMP:
            break;

        case DNLD_CTRL_PREPARE_FIRMWARE:
            // tell bootloader to upgrade firmware
            // reboots and does not return.
            CresnetBootloaderRequest(1, GetStreamFromPacket(packet[0]));
            break;

        case DNLD_CTRL_BLK_COMPLETE:
#ifdef STR91X_IAR
          // at end of block check if we need to write cached bootloader vector area
          // we can only write it at the end of the block, when control system
          // is waiting for "ready for d/l" response
            WriteBootloaderVectorBlockIfNeeded();
#endif // STR91X_IAR
            if (NULL != pfCresnetDataXferBlockComplete)
            {
                // Invoke whatever message-handler was set by application.
                //  It should send a reply ("Ready" or "Abort") to the host.
                pfCresnetDataXferBlockComplete(GetStreamFromPacket(packet[0]));
            }
            else
            {
                // Default behavior, but potential for ERROR: Automatically
                //  reply to host without involving/notifying the application.
                CresnetSlaveSendReadyForXfer(CresnetSlave.blockBytes, GetStreamFromPacket(packet[0]));
            }
            break;

        case DNLD_CTRL_ABORT:
            CresnetDataXferAbort(GetStreamFromPacket(packet[0]));
            EnableNVLWrites();
            break;

        default:
            break;
    }
}

// do we really need all the reserved joins?
#define TOUCH_SETTABLE_ID_BUTTON    17915

/**
 * \author      Pete McCormick
 * \date        11/1/2005
 * \return      void
 *
 * \brief       Process digital packets
 *
 * \param       bStream
 * \param       value
 * \param       join
 * \param       source
 */
static void CresnetSlaveProcessDigital(UINT8 bStream, UINT32 join, UINT32 value, UINT32 source)
{
    switch(join)
    {
        case TOUCH_SETTABLE_ID_BUTTON:
            PPNSetupLED((UINT8)value);
            break;
        default:
            CresnetSlaveProcessInternalDigital(bStream, join, value, source);
            break;
    }
}

/**
 * \author      Pete McCormick
 * \date        11/1/2005
 * \return      void
 *
 * \brief       CresnetSlaveApplyIndirectTextChange
 *
 * \param       bStream
 * \param       value
 * \param       slot
 * \param       source
 */
void CresnetSlaveApplyIndirectTextChange(UINT8 *command, UINT32 command_length, UINT32 source, UINT32 slot)
{
//    UINT32 status;
    UINT32 sub_command_length;
    //struct join_map_entry_st *serial_list;
    UINT32 text_field;
    UINT8 *field_pointer;
    UINT32 digit_count;
    UINT8 first_char;
    UINT8 bStream;
    UINT32 join = 0;
    char * p;

    // get stream
    bStream = GetStreamFromPacket(*command);
    command += 3;  // skip rest of header

    /* Chop up the string and process each seperately */
    while(command_length)
    {
        for (sub_command_length = 1;
            sub_command_length <= command_length;
            sub_command_length++)
        {
            if (command[sub_command_length - 1] == 0x0D)
                break;
        }

        /* we now have a sub command length */

        /* extract the signal number */
        field_pointer = command;

        /* check for initial # or @ symbol */
        first_char = *field_pointer++;
        if ( (first_char != '#') && (first_char != '@') )
        {
            break;
        }

        /* get the field number */
        text_field = 0;
        digit_count = 0;
        while(isdigit(*field_pointer))
        {
            text_field *= 10;
            text_field += *field_pointer++ - '0';
            digit_count++;
        }

        // text_field now has the serial join number
        // field_pointer points to the comma before the actual string data
        field_pointer++;                // skip ','

        if(*field_pointer != '\r')
        {
            p = (char *)field_pointer;
            while(*p != '\r')
            {
                p++;
            }
            // replace \r with NULL
            *p = 0;

            join = (UINT16)text_field;

            if( slot != TOP_LEVEL_SLOT_0 )//if sub-slot
            {
               join = ApplySubSlotMaskToJoinNumber( slot, join );

               //Check for an error
               if ( !IsSubSlottedJoin(join) )
                   continue;
            }

            CresnetSlaveProcessInternalSerial(bStream, join, (char *)field_pointer,
                p-(char *)field_pointer, source);
        }
        /* skip past the partial command */
        command += sub_command_length;
        command_length -= sub_command_length;
    }
}

/**
 * \author      Pete McCormick
 * \date        11/1/2005
 * \return      void
 *
 * \brief       CresnetSlaveRxParseMultiSerial
 *
 * \param       bStream
 * \param       value
 * \param       slot
 * \param       source
 */
void CresnetSlaveRxParseMultiSerial(UINT8 *packet, UINT32 source, UINT32 maxlen,
        UINT32 slot, UINT16 packlen)
{
    UINT32 join=0;
    //UINT8 *DataPtr;

    if ( *(packet + 3) & 0x80 )  // check for control messages.
    {
        if ( *(packet + 4) == 0)  // connect/configure message
        {
            // if serial control port present, send it the configuration
            // otherwise redirect console to Cresent
            if( !ProductHasSerialPort((UINT8 *)(packet+3), slot ))
            {
                CresnetConsole->CresnetConsoleRedirectConnect(packet);
            }
        }
        else if ( *(packet + 4) == 1 )  // disconnect message
        {
            CresnetConsole->CresnetConsoleRedirectDisc(packet);
        }
    }
    else if ( CresnetConsole->CresnetConsoleRedirectIsActive() )   // data only packet
    {
        CresnetConsole->CresnetConsoleRedirectData(packet, packlen);
    } // if (control/config message)
    else
    {
        // got some data here from control system due to a JOINSETSERIAL command
        // Multi-channel serial packet:
        //[0]  [1]   [2]  [3]       [4]       [5]     [6]   [7]
        //<id> <cnt> <12> <flag+ch> <data>    .       .     .

       join = packet[3];

       if( slot != TOP_LEVEL_SLOT_0 )//if sub-slot
       {
          join = ApplySubSlotMaskToJoinNumber( slot, join );

          //Check for an error
          if ( !IsSubSlottedJoin(join) )
              return;
       }

       if(packlen+2 < maxlen)
       {
           packet[packlen+2] = 0;
       }

       CresnetSlaveProcessInternalSerial(GetStreamFromPacket(packet[0]), join, (char *)&packet[4],
                   packlen-2, source);
    }
}

// temp stubs

/*
 * \author      Aleksandr Baranov
 * \date        04/06/2011
 * \return      void
 *
 * \brief       Synchronize RTC with Control System
 *
 * \param
 * \param
 * \param
 * \param
*/
void settime(short hour,short minute,short second)
{
    if(SetTimeforRTC != NULL)
    {
        SetTimeforRTC(hour, minute, second );
    }
}
void setdate(short month,short day,short year)
{
    if(SetDateforRTC != NULL)
    {
        SetDateforRTC( month, day, year);
    }
}
/**
 * \author      Pete McCormick
 * \date        11/1/2005
 * \return      void
 *
 * \brief       CresnetSlaveRxParseSched
 *
 * \param       bStream
 * \param       value
 * \param       slot
 * \param       source
 */
void CresnetSlaveRxParseSched(UINT8 *packet, UINT32 source, UINT32 maxlen, UINT32 slot)
{
    UINT16 hour,minute,second,month,day,year;

    // NOTE: This currently is stream independent

    switch ( *(packet+3) )
    {
        case 0:     // ON EVENT
        case 1:     // OFF EVENT
        case 2:     // PUSH EVENT
        case 0x10:  // NOT IN LIST
        case 0x11:  // END OF LIST
        case 0x12:  // EMPTY LIST ?
            break;
        case 0xC:  /* set time (values are BCD) */
            hour   =(UINT16)(bcdval(*(packet+4)));
            minute =(UINT16)(bcdval(*(packet+5)));
            second =(UINT16)(bcdval(*(packet+6)));
            settime(hour,minute,second);
            break;
        case 0xD:  /* set date */
            month  =(UINT16)(bcdval(*(packet+4)));
            day    =(UINT16)(bcdval(*(packet+5)));
            year   =(UINT16)(bcdval(*(packet+6)));
            setdate(month,day,year);
            break;
        case 0xE:   /* set time and date */
            hour   =(UINT16)(bcdval(*(packet+4)));
            minute =(UINT16)(bcdval(*(packet+5)));
            second =(UINT16)(bcdval(*(packet+6)));
            settime(hour,minute,second);
            month  =(UINT16)(bcdval(*(packet+7)));
            day    =(UINT16)(bcdval(*(packet+8)));
            year   =(UINT16)(bcdval(*(packet+9)));
            setdate(month,day,year);
            break;
        default:
            break;
    }
}

// move this elsewhere
UINT8 SRecordCalcChecksum(UINT8 *record, UINT32 byteCnt)
{
    UINT8 csum;
    UINT32 i;

    // + 4 for address bytes, +1 for length byte
    csum = byteCnt + 5; // assumes that transferred record is always S3
    for(i=0; i<byteCnt+4; i++)
    {
        csum += record[i];
    }
    return ~csum;
}

/**
* \author    Larry Salant
* \brief     Sends an abort transfer packet
* \date      8/26/2008
* \param     none
* \return    void
* \retval    none
**/
void CresnetFlashLoaderSendAbortXfer(void)
{
  UINT8 packet[5];
  UINT8 * p;

  p = packet;

  // build the response packet
  *p++ = CNET_ID_CTRLSYS;   /* destination ID */
  *p++ = 2;   // packet length
  *p++ = CNET_GLOBAL;
  *p++ = DNLD_CTRL_ABORT;

  // send message
  CresnetSlaveEnqueueOutgoingPacket(0, packet, 4, NET_DEST_CRESNET);
}


/**
 * \author      Pete McCormick
 *
 * \date        3/12/2008
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Handle some S-Record data from the Control System
 *
 * \param       packet
 * \param       source
 * \param       maxlen
 * \param       slot
 *
 * \note       This currently is stream independent
 *
 * \warning     S3 records only? (4-byte addresses)
 *
 */

void CresnetSlaveRxParseSrecord(UINT8 *packet, UINT32 source, UINT32 maxlen, UINT32 slot)
{
    //         0    1     2           3      4      5      6      7
    // format: <id> <len> <type==0xd> <addr> <addr> <addr> <addr> <data> ... <csum>

    UINT8 len = packet[1];
    UINT32 byteCnt = len - 1 - 1 - 4; // -1 for type, -1 for csum byte, -4 for addr
    UINT32 addr = MAKEUINT32(MAKEUINT16(packet[6],packet[5]),MAKEUINT16(packet[4],packet[3]));
    UINT8 * pData = &packet[7];
    UINT8 csum = packet[len+1]; // last byte is the checksum
    UINT8 calcCsum;
    BOOL csumValid;
    UINT32 flashId;
    UINT32 blockSize;

    calcCsum = SRecordCalcChecksum(&packet[3], byteCnt);
    csumValid = (calcCsum == csum) ? true : false;

    if(addr == 0xDEADBE )//first S-record
    {
        blockSize   = packet[10];  //send card id in this parameter
        flashId     = packet[14];  //which flash to program

        DisableNVLWrites();
        if ( !CresnetDataXferStart(flashId, &blockSize, GetStreamFromPacket(packet[0]), 0))
        {
            //abort cresnet data transfer
            CresnetFlashLoaderSendAbortXfer();
        }

    }
    else //successive S-records
    {
        CresnetDataXferData(addr, pData, byteCnt, csumValid, GetStreamFromPacket(packet[0]));//this should be next address, data etc..
    }
}
/**
 * \author      Andrew Salmon
 * \date        10/06/2008
 * \brief       Applies the subsot mask to joins on subslot
 * \return      UINT32
 * \retval      masked join number or same join number id slot not supported
 * \param       slot - slot number
 * \param       join - 0-based join number
 */
UINT32 ApplySubSlotMaskToJoinNumber( UINT8 slot, UINT32 join )
{
    //Verify the slot
    if ( !IsValidSubSlotNumber(slot) )
    {
        //Log a warning and return the join
        DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_UNSUPPORTED_SLOT_ID,slot);
        return join;
    }

    //Clear the subslot from the join
    join &= SUBSLOT_JOIN_MASK;

    //Set the subslot bits
    join = SET_SUBSLOT_BITS(join, slot);

    return join;
}

/**
 * \author      Yun Mao
 * \date        4/16/2007
 * \return      void
 * \brief       Processes joins received from the switch's central
 *              processor
 * \param       slot : which slot it is
 * \param       packet: A pointer to the packet data
 * \param       source :where it came from
 * \param       maxlen : packet length
 * \note        The adding/subtracting 1 was being done on the Touch
 *              Panels. On the Touch Panels, join number 0 was
 *              not being used because of compatibility issues
 *              with VTPRO. This is not required on the DM
 *              Cards. So the cresnet code need not add/subtract
 *              1 for the digital, analog, and serial joins.
 *              Also, the cresnet code was updated to use Type
 *              15 for the serial joins, type 14 for analog
 *              joins.
 * \warning
 */
BOOL CresnetSlaveRxParse(UINT8 *packet, UINT32 source, UINT32 maxlen, UINT32 slot)
{
    BOOL bFreePacket = TRUE;  //Assume the incoming packet needs to be freed
    UINT8 packlen;
    UINT8 packtype;
    UINT32 join = 0;
    UINT16 digital_value = 0;
    UINT16 anumb = 0;
    UINT16 analog_value = 0;
    UINT8  serial_flag;
    UINT8  tempValue = 0;
#ifdef  RCB_ANALOG_SUPPORT
    UINT32 time = 0;
#endif

    //Variables used for multi-join packets
    UINT8 i;
    UINT16 sTotalLength;

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
    UINT8 bDMNetIdx;
    CDMSwitchCard *pCardPtr     = NULL;
#endif

#ifndef REMOVE_DM_ENCAP_HELPER
    DmEncapChecksumHelper* pEncapHelper = 0;
#endif
    // packet validate and token passing now handled in ValidateLocalCresnetMsg

    //id = *packet;
    packlen =(*(packet+1) & 0xff);      /* temp variables for easier debug */
    packtype=*(packet+2);

    if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_RX_IDX))
        DmConsoleHexDumpPacket(packet, packet[1] + 2, "CnetRx:", 1);

    // Check for packet rerouting before going through the rest of the parsing
    if (pfPreCnetSlaveParsePktIntercept)
    {
        // Any unhandled packets should go through the rest of the parser
        if (pfPreCnetSlaveParsePktIntercept(packet, packlen, packtype, source))
            return bFreePacket;
    }

    if ( packlen == 1 )
    {   /* special short digital on/off command */
        digital_value = !(*(packet+2) & 0x80);

        join = (UINT16)(*(packet+2) & 0x7F);

        if ( slot != TOP_LEVEL_SLOT_0 )//if sub-slot
        {
            join = ApplySubSlotMaskToJoinNumber( slot, join );

            //Check for an error
            if ( !IsSubSlottedJoin(join) )
                return bFreePacket;
        }

        if (CresnetSlave.debug & 1)
        {
            DmConsolePrintf("CNET: rx digital(%d)=%u \r", join, digital_value);
        }
        CresnetSlaveProcessDigital(GetStreamFromPacket(packet[0]), join, digital_value, source);
        return bFreePacket;
    }

    switch ( packtype )
    {     /* check the command type */
        case CNET_DIGITAL:           /* long form digital on/off */
        case INT_DIGITAL:           /* local digital */
            //Calculate the total packet length
            sTotalLength = packlen + 2;

            //Start at the beginning
            i = 3;
            while ( (i+2) <= sTotalLength )
            {
                //[0]  [1]   [2]   [3]   [4]
                //<id> <cnt> <00> <LSB> <MSB | digital value>
                digital_value = !(packet[i+1] & 0x80);
                join = (((UINT16)(packet[i+1] & 0x7F)) << 8) + ((UINT16)packet[i]) ;


#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT

            if(GetDMNetIdxFromSlot(&bDMNetIdx, slot, CtrlrConfig.bMaxInputSlotNum, 
				   CtrlrConfig.bMaxOutputSlotNum)==0)
		{
		    pCardPtr = g_DmNetDiscoverDevice->m_pCards[bDMNetIdx];

		    if (!pCardPtr->m_pDMNetMapper)
			slot = bDMNetIdx+1;
		}
#endif

                if ( slot != TOP_LEVEL_SLOT_0 )//if sub-slot
                {
                    join = ApplySubSlotMaskToJoinNumber( slot, join );

                    //Check for an error
                    if ( !IsSubSlottedJoin(join) )
                        return bFreePacket;
                }

                CresnetSlaveProcessDigital(GetStreamFromPacket(packet[0]), join, digital_value, source);

                if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_RX_IDX))
                    DmDebugPrintf("CnetRx: Digital %d = 0x%02x\r", join, digital_value);

                //Increment to next join
                i+=2;
            }
            break;

        case CNET_ANALOG:
        case INT_ANALOG:
            if (packlen == 3)
            {
                //[0]  [1]    [2]  [3]    [4]
                //<id> <cnt> <01> <chan> <val>
                /* short format */
                anumb = packet[3];
                analog_value = packet[4];
            }
            else if (packlen == 4)
            {
                //[0]  [1]   [2]   [3]    [4]     [5]
                //<id> <cnt> <14> <chan> <valhi> <vallo>
                /* long value format */
                anumb = packet[3];
                analog_value = MAKEUINT16(packet[5],packet[4]);
            }
            else if (packlen == 5)
            {
                //[0]  [1]   [2]   [3]       [4]       [5]     [6]
                //<id> <cnt> <14> <chan hi> <chan lo> <valhi> <vallo>
                /* long number and long value */
                anumb = MAKEUINT16(packet[4],packet[3]);
                analog_value = MAKEUINT16(packet[6],packet[5]);
            }

            join = anumb;

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT

            if(GetDMNetIdxFromSlot(&bDMNetIdx, slot, CtrlrConfig.bMaxInputSlotNum, 
				   CtrlrConfig.bMaxOutputSlotNum)==0)
		{
		    pCardPtr = g_DmNetDiscoverDevice->m_pCards[bDMNetIdx];

		    if (!pCardPtr->m_pDMNetMapper)
			slot = bDMNetIdx+1;
		}
#endif


            if ( slot != TOP_LEVEL_SLOT_0 )//if sub-slot
            {
                join = ApplySubSlotMaskToJoinNumber( slot, join );

                //Check for an error
                if ( !IsSubSlottedJoin(join) )
                    return bFreePacket;
            }

            if (CresnetSlave.debug & 1)
            {
                DmConsolePrintf("CNET:rx analog(%d)=%u \r", join, analog_value);
            }
            CresnetSlaveProcessInternalAnalog(GetStreamFromPacket(packet[0]), join, analog_value, source);

            if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_RX_IDX))
                DmDebugPrintf("CnetRx: Analog1 %d = 0x%04x\r", join, analog_value);
            break;

        case PORT_FORCE:
            /* check for indirect text message */
            if ((packet[3] == '#') || (packet[3] == '@'))
            {
                CresnetSlaveApplyIndirectTextChange(packet,(UINT32)(packlen-1), source, slot);
            }
            break;

        case CNET_COMMAND:

            CresnetSlaveRxParseCmd(packet, source, maxlen, slot);
            break;

        case CNET_GLOBAL:

            CresnetSlaveRxParseGlobal(packet, source, maxlen, slot);
            break;

        case SCHED:

            CresnetSlaveRxParseSched(packet, source, maxlen, slot);
            break;

        case SRECORD:

            CresnetSlaveRxParseSrecord(packet, source, maxlen, slot);
            break;

        case CNET_MULTISERIAL :
            CresnetSlaveRxParseMultiSerial(packet, source, maxlen, slot, packlen);
            break;


        case GATEWAY_MSG:
#ifdef NEEDED
            CresnetSlaveRxParseGateway(packet, source, maxlen, slot, packlen);
#endif //NEEDED
            break;

        case ANALOG_SYMMETRIC:
            //Calculate the total packet length
            sTotalLength = packlen + 2;

            //Start at the beginning
            i = 3;
            while ( (i+4) <= sTotalLength )
            {
                //[0]  [1]   [2]   [3]       [4]       [5]     [6]
                //<id> <cnt> <14> <chan hi> <chan lo> <valhi> <vallo>
                anumb = MAKEUINT16(packet[i+1],packet[i]);
                analog_value = MAKEUINT16(packet[i+3],packet[i+2]);

                join = anumb;

                if ( slot != TOP_LEVEL_SLOT_0 )//if sub-slot
                {
                    join = ApplySubSlotMaskToJoinNumber( slot, join );

                    //Check for an error
                    if ( !IsSubSlottedJoin(join) )
                        return bFreePacket;
                }

                CresnetSlaveProcessInternalAnalog(GetStreamFromPacket(packet[0]), join, analog_value, source);

                if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_RX_IDX))
                    DmDebugPrintf("CnetRx: Analog14 %d = 0x%04x\r", join, analog_value);

                //Increment to next join
                i+=4;
            }
            break;

        case CRESNET_SERIAL:
#ifdef DM_V24
        case UNICODE_SERIAL_IO:
#endif
            //[0]  [1]   [2]  [3]       [4]       [5]     [6]
            //<id> <cnt> <15> <chan hi> <chan lo> <flags> [data...]
            // null-terminate
            if (packlen+2 < maxlen)
            {
                packet[packlen+2] = 0;
            }

            join = MAKEUINT16(packet[4],packet[3]);

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT

            if(GetDMNetIdxFromSlot(&bDMNetIdx, slot, CtrlrConfig.bMaxInputSlotNum, 
				   CtrlrConfig.bMaxOutputSlotNum)==0)
		{
		    pCardPtr = g_DmNetDiscoverDevice->m_pCards[bDMNetIdx];

		    if (!pCardPtr->m_pDMNetMapper)
			slot = bDMNetIdx+1;
		}
#endif

            if ( slot != TOP_LEVEL_SLOT_0 )//if sub-slot
            {
                join = ApplySubSlotMaskToJoinNumber( slot, join );

                //Check for an error
                if ( !IsSubSlottedJoin(join) )
                    return bFreePacket;
            }

            serial_flag = packet[5];
            if (CresnetSlave.debug & 1)
            {
                int idx;
                DmConsolePrintf("rx serial length = %d \r", packlen-4);
                for (idx = 0; idx<(packlen-4); idx++)
                    DmConsolePrintf("rx serial(%d)=0x %x\r", join, (char *)&packet[6+idx]);
            }

            CresnetSlaveProcessInternalSerial(GetStreamFromPacket(packet[0]), join, (char *)&packet[6],
                                              packlen-4, source,serial_flag);

            if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_RX_IDX))
            {
                for (int idx = 0; idx<(packlen-4); idx++)
                {
                    tempValue = (UINT8)packet[6+idx];
                    DmDebugPrintf("CnetRx: serial(%d)=0x %x\r", join, tempValue);
                }
            }

            break;

        case IR_PACKET:
            CresnetSlaveProcessIr( packet, packlen, source );
            break;

        case PORT_REGISTRATION:
        case SET_VERSIPORT:
        case INTERNAL:
            CresnetSlaveProcessInternalCommand( packet, packlen, source );
            break;

        case CRESNET_ENCAPSULATED:

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
            // Check to see if these are the generic configuration Type 1C packets - If so they should not make it to the logic
            if (DMMidpointControllerHandleDMDeviceConfigurationPacket((CREST_ALL_PACKETS *)packet))
                break;
#endif

            // if not connected to another DM device, process packet locally 
            if ( !pDMNetMapper || (pDMNetMapper && !pDMNetMapper->SendEncapToEndpoint(packet, GetStreamFromPacket(packet[0])%TOTAL_NUMBER_OF_DM_MASTERS)))
            {
                //[0]  [1]   [2]  [3]
                //<id> <cnt> <20> <slot>
                slot = packet[3];
                if ( !IsValidSubSlotNumber(slot) )
                {
                    //Log error as a warning
                    DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_UNSUPPORTED_SLOT_ID,packet[3]);
                    return bFreePacket;
                }
                if (packet[5] == CRESNET_ENCAPSULATED)
                {
                    // too many levels of encapsulation, get out!
                    // We cannot handle more than 1 level of encapsulation
                    DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_BAD_NEST_ENCAPSULATION,0);
                    return bFreePacket;
                }

                // warning: recursive call!
                CresnetSlaveRxParse(&packet[3], source, maxlen, slot);
            } // end if no DmHosts
            break;

        case RCON_PKT:
            //Bugzilla 80918 fix
            //Check if we should process RCON packets in a separate timer to prevent the parser task from being blocked
            if ( CresnetRconTimer::IsTimerCreated() )
            {
                if ( CresnetRconTimer::Add(packet) )
                {
                    //If we successfully added the packet to the CresnetRconTimer, do not free it
                    bFreePacket = FALSE;
                }
                else
                {
                    //Log an error since the CresnetRconTimer failed and throw away the packet
                    DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_CNET, ERR_CNET_SLAVE_RCON_TIMER_ADD_PACKET_FAILED, 0);
                    bFreePacket = TRUE;
                }
            }
            else
            {
                //Old method of waiting on the CresnetConsole which is unsafe
                if ( CresnetConsole )
                {
                    CresnetConsole->CresnetConsoleProcessRcon(packet, MAX_CONSOLE_WAIT);
                }
            }
            break;

            // handle PNP packets
        case PNP_SET_STATE:
        case PNP_SET_BY_CNET_ID:
        case PNP_SET_BY_RANDOMIZE:
        case TEMP_NET_ID:
        case NEW_NET_ID:
            if (pfPNPRxParse)
                (*pfPNPRxParse)(packet, (UINT8)(packtype & 0xff));
            break;

        case TJI_REQUEST:
        case CAPABILITY_REQUEST:
        case EXTENDED_CAPABILITY_REQUEST:
            //[0]  [1]   [2]  [3]    [4]     [5]
            //<id> <cnt> <30> <type> <Flags> <Transaction #>
            // type is reserved, Flags are for symmetry with response packet
            // <Transaction#> is a unique byte id.  The response to this will have the same <Transaction#> in it.

            GetState(GetStreamFromPacket(packet[0]))->CresnetCommandMemorize(packet, source, slot);
            break;

        case DM_DEV_UPDATE_JOIN_REQUEST_CMD:
            //[0]  [1]   [2]  [3]
            //<id> <cnt> <36> <flags>
            if ( packet[3] == DM_DEV_REFRESH_DMNET_JOINS )
            {
                CresnetDmJoinRefresh(GetStreamFromPacket(packet[0]));
            }
            break;

            // Digital Media Control Flow Direction Message
            // <id> <cnt> <32> <Flags>
            // This packet is sent by the switch firmware to all the DM Links under the DM Input Card requesting to
            // reverse/normalize the control flow direction.  All Repeaters / DGE / WP / DM Output Card will accept this message,
            // act on it and pass it on. No DM devices will respond to this message. If a DM Output Card receives this message,
            // it should just pass this message onto the DM Switch firmware.
            // The <id> will always be a BroadCast ID (0xFF).
            // <Flags> are defined as: 01: Reverse Control Flow; 02: Normal Control Flow
        case DM_DEV_CONTROL_FLOW_DIR_CMD:
            // this is processed when the Master port first receives the packet
            // just need to re-broadcast it.
            break;

            // Digital Media Control Flow Reject Message
            // <id> <cnt> <33> <Flags>
            // For 2 DM switches directly connect.  The output card takes precedent over the input card (ie normal flow).
            // This packet is only sent by the switch firmware to the DM Output Card in response to receiving the
            // Digital Media Control Flow Direction Message with the flags set to Reverse Control Flow from the DM Output Card.
            // Any DM devices which receive this message should just pass this  no action should be taken in response to this message.
            // This message will only be processed by the second DM Switch firmware. The switch on receipt of this message will send out
            // a Digital Media Control Flow Direction Message with the flags set as Normal Control Flow.
            // The <id> will always be a BroadCast ID (0xFF).
            // <Flags> are defined as: 01: Reverse Control Flow Rejected
        case DM_DEV_CONTROL_FLOW_REJECT_CMD:
            break;

// note: due to ram and code space limitation for DM products, RCB tx/rx is added as a conditional compile
#ifdef  RCB_ANALOG_SUPPORT
        case RCB_CMD:
            //Calculate the total packet length
            sTotalLength = packlen + 2;

            //Start at the beginning
            i = 3;

            time = MAKEUINT32 (MAKEUINT16(packet[i+3],packet[i+2]), MAKEUINT16(packet[i+1],packet[i+0]));

            i += 4; // skip over the time field
            while ( (i+5) <= sTotalLength )
            {
                //[0]   [1]   [2]  [3..6]    [7..8]          [9..10]               [11]          [12..13]         [14..15]             [16]
                //<id> <cnt> <1E> <32b time> [<16b channel 1><16b Terminus level 1><8b flags 1>][<16b channel N><16b Terminus level N><8b flags N>]
                anumb = MAKEUINT16(packet[i+1],packet[i]);
                analog_value = MAKEUINT16(packet[i+3],packet[i+2]);
                UINT8 flag = packet[i+4];
                join = anumb;

                if ( slot != TOP_LEVEL_SLOT_0 )//if sub-slot
                {
                    join = ApplySubSlotMaskToJoinNumber( slot, join );

                    //Check for an error
                    if ( !IsSubSlottedJoin(join) )
                        return bFreePacket;
                }

                CresnetSlaveProcessInternalRCB(GetStreamFromPacket(packet[0]), join, analog_value, time, flag, source);

                if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_RX_IDX))
                    DmDebugPrintf("CnetRx: RCB %d = 0x%04x\r", join, analog_value);

                //Increment to next join
                i+=5;
            }
            break;

        case CLX_CMD:
            //  <id> <cnt> <1D> <24b time> [<8b channel 1><8b Terminus level 1>[<8b channel N><8b Terminus level N>]

            //Calculate the total packet length
            sTotalLength = packlen + 2;

            //Start at the beginning
            i = 3;

            time = MAKEUINT32 (MAKEUINT16(packet[i+2],0), MAKEUINT16(packet[i+1],packet[i+0]));

            i += 3; // skip over the time field
            while ( (i+2) <= sTotalLength )
            {
                //[0]   [1]   [2]  [3..5]    [6]            [7]                    [8]           [9]
                //<id> <cnt> <1D> <24b time> [<8b channel 1><8b Terminus level 1>][<8 channel N><8 Terminus level N>]
                anumb = packet[i];
                analog_value = packet[i+1];
                join = anumb;

                if ( slot != TOP_LEVEL_SLOT_0 )//if sub-slot
                {
                    join = ApplySubSlotMaskToJoinNumber( slot, join );

                    //Check for an error
                    if ( !IsSubSlottedJoin(join) )
                        return bFreePacket;
                }

                CresnetSlaveProcessInternalCLX(GetStreamFromPacket(packet[0]), join, analog_value, time, source);

                if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_RX_IDX))
                    DmDebugPrintf("CnetRx: CLX %d = 0x%04x\r", join, analog_value);

                // Increment to next join
                i+=2;
            }
            break;
#endif

#ifndef REMOVE_DM_ENCAP_HELPER
        case DM_SEQUENCE_REQUEST_PACKET_CMD:
            pEncapHelper = GetDmEncapChecksumHelper(GetStreamFromPacket(packet[0]));
            if ( pEncapHelper )
            {
                //Process the sequence request
                pEncapHelper->ProcessSequenceRequest(packet);
            }
            break;

        case DM_SEQUENCE_RESPONSE_PACKET_CMD:
            pEncapHelper = GetDmEncapChecksumHelper(GetStreamFromPacket(packet[0]));
            if ( pEncapHelper )
            {
                //Process the sequence response
                pEncapHelper->ProcessSequenceResponse(packet);
            }
            break;

        case DM_LONG_REACH_PACKET_CMD:
            // This packet type is still used for ForceStandardMode
            pEncapHelper = GetDmEncapChecksumHelper(GetStreamFromPacket(packet[0]));
            if ( pEncapHelper )
            {
                //Process the long reach packet
                pEncapHelper->ProcessLongReachPacket(packet);
            }
            break;
#endif

        case PKT_MESSAGE: // display message on the screen
            if (pfProcessPacketType_2C)
                pfProcessPacketType_2C(packet);
            break;

        default:
            {
                // Check if we should handle the unrecognized packet
                if (pfPostCnetSlaveParsePktProcess)
                    pfPostCnetSlaveParsePktProcess(packet, packlen, packtype, source);
            }
//            DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_RX_UNKNOWN_PKT, packtype & 0xff);
            break;
    }

    if (pDMNetMapper)
    {
        // forward broadcast packets to any "downstream" devices
        pDMNetMapper->SendBroadcastToEndpoint(packet);
    }

    return bFreePacket;
}

// Functions that work for both uart cresnet and cip
// get the "Cresnet" prefix
// 1st byte of packet is actually the source

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       puts pointer to a buffer containing a packet into the
 *              queue for the cresnet slave input parser task.
 * \return      none
 * \retval      void
 * \param       param - pointer to the cresnet packet
 */
UINT32 CresnetSlaveAcceptIncomingPacket(UINT8 *packet)
{
    UINT32 flags = 0;

    if(OsQueue(CresnetSlave.inQueue, &packet, OS_WAIT_FOREVER ) == 1)
    {
        flags |= OS_ISR_FLAG_TASK_WOKEN;
    }
    return flags;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       queue task to handle parsing incoming msgs
 * \return      none
 * \retval      void
 * \param       param - pointer to the cresnet packet
 */
void CresnetSlaveInParserTask(UINT32 param)
{
    UINT8 * packet = (UINT8*)param;

    if (CresnetSlave.debug)
    {
        // here +2 is the offset for the length, since we shifted
        // down by 1 in order to keep the source in packet[0]
        DmConsoleHexDumpPacket(packet, (UINT16)packet[2]+2, "Cresnet RX:", 1);
    }

    // parsing skips byte where we put the source
    // invoke the common cresnet parser
    BOOL bFreePacket = CresnetSlaveRxParse(&packet[1], packet[0], CNET_MAX_MSG_SIZE, TOP_LEVEL_SLOT_0);

    // free the buffer used for this command
    if ( bFreePacket && MemMgr)
    {
        MemMgr->FreeBlock(packet);
    }
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       queue task to handle parsing incoming msgs
 * \return      none
 * \retval      void
 * \param       param - pointer to the cresnet packet
 */
void CresnetSlaveInCombinedPacketParserTask(UINT32 param)
{
    COMBINED_CNET_PKT* pComPkt = (COMBINED_CNET_PKT*)param;
    UINT8 TotalLen = pComPkt->Length;
    UINT8 Source = pComPkt->Source;

    // Print out the packet for debugging if needed
    if (CresnetSlave.debug)
      DmConsoleHexDumpPacket((UINT8*)pComPkt, TotalLen, "Cresnet RX:", 1);

    GENERIC_CREST_PACKET* pCnetPkt = &pComPkt->CnetPkt;

    // Packet must have the cresnet header at minimum
    while (TotalLen > CRESNET_PACKET_HEADER_LENGTH)
    {
      UINT8 PktLen = pCnetPkt->len + CRESNET_PACKET_HEADER_LENGTH;

      // Check to make sure we really have this smany bytes left. Safety check!
      // If the packet len is good, pass it to the parser
      if (TotalLen >= PktLen)
        CresnetSlaveRxParse((UINT8*)pCnetPkt, Source, PktLen, TOP_LEVEL_SLOT_0);
      else
        break; // Not enough data to make a real packet. Something is wrong!

      pCnetPkt = (GENERIC_CREST_PACKET*)((UINT8*)pCnetPkt + PktLen);
      TotalLen -= PktLen;
    }

    // free the buffer used for this command
    if (MemMgr)
      MemMgr->FreeBlock((UINT8*)pComPkt);
}

void CresnetSlaveSetUploadBlockSize(UINT16 blockSize)
{
    CresnetSlave.blockBytes = blockSize;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       creates the Cresnet slave input task, its queue and lock for buffer
 * \return      none
 * \retval      void
 * \param       none
 * \note      Jan 13 2009, R.H: Changed analog joins size to 16 bit.
 */
void CresnetSlaveInit(UINT8 NumStream, UINT16 NumDigitals,
                      UINT16 NumAnalogs, UINT16 NumSerials,
                      UINT8 NumCommands, UINT32 blockSize /* = 0 */, UINT8 bPackJoinLimit, /* = 0 */
                      UINT8 bQueueSize /* = DM_DEFAULT_CNET_SLAVE_QUEUE_SIZE */,
                      UINT16 additionalstacksize /* = 0*/, BOOL CombinedPackets /*= FALSE*/, 
                      UINT8 bRconTimerBufferCount /* = 0 */ )
{
    UINT32 param;
    CresnetJoinState **pCurrent;

    //Create the CresnetRconTimer thread
    if ( bRconTimerBufferCount )
    {
        CresnetRconTimer::Create(bRconTimerBufferCount);
    }

    //Validate the queue size
    if ( !bQueueSize )
        bQueueSize = DM_DEFAULT_CNET_SLAVE_QUEUE_SIZE;

    // will crash if can't create task
    // periodic, queue, param
    if (!CombinedPackets)
    {
      OsCreateNamedPriorityTask(CresnetSlaveInParserTask, 0, bQueueSize, &param,
                                CresnetSlave.extra, "CrS",CRESNET_SLAVE_TASK_PRIORITY, OsGetDefaultStackVarCnt()+additionalstacksize);
    }
    else
    {
      OsCreateNamedPriorityTask(CresnetSlaveInCombinedPacketParserTask, 0, bQueueSize, &param,
                                CresnetSlave.extra, "CrS",CRESNET_SLAVE_TASK_PRIORITY, OsGetDefaultStackVarCnt()+additionalstacksize);

    }

    CresnetSlave.inQueue = param;
    CresnetSlave.disableIso8859 = 1;
    CresnetSlave.blockBytes = blockSize;

    // create lock for output buffer
    OsCreateLock(&CresnetSlaveLockId);

    // remember how many streams
    CresnetSlave.NumStreams = NumStream;

    // Only create this if we have any streams. This will be the case for the IRController
    if (NumStream) {

      // init the join state.  This is Card specific since the number of joins and streams may vary
      // create an array of pointers to the Join state objects
      pJoinState = (CresnetJoinState **)malloc(NumStream * sizeof(CresnetJoinState *));
      pCurrent = pJoinState;

      // for each stream
      for (int iStream = 0; iStream < NumStream; iStream++)
      {
        // create a join state map with the appropriate number of joins
        *pCurrent = new CresnetJoinState(iStream, NumDigitals, NumAnalogs, NumSerials, NumCommands, bPackJoinLimit);
        ++pCurrent;
      }

      // this should be a separate task but for now run on timer
      //TBD create task - when memory available
      OsStartTimer(DM_CRESNET_SLAVE_SEND_PENDING_TIMER_MS,
                   (void *)CresnetSlaveSendPending,
                   OS_TIMER_PERIODIC);
    }

    // set the default output destination
    //TBD - set based on device
    CresnetSlave.dest = NET_DEST_CRESNET;

    // create class for Cresnet Console redirection
    CresnetConsole = new CCresnetConsole();

    // must register with SetCnetCustomFunctionPointer
    m_pfCnetCustom = NULL;

}
/**
 * \author    Larry Salant
 * \brief     gets a pointer to the object for the specified stream
 * \date      2/5/2008
 * \param     RxId - output number of the Receiver
 * \return    void
 * \retval    none
**/
CresnetJoinState *GetState(UINT8 bStream)
{

  if (bStream >= CresnetSlave.NumStreams)
  {
    DmSystemError(DM_ERROR_LEVEL_WARNING,DM_ERROR_SUBSYS_CNET,ERR_CNET_INVALID_STREAM,bStream);

    bStream = 0;
  }

  return (*(pJoinState+bStream));
}

void CresnetInitDefaults(void)
{
}

void CresnetParamsRestore(void)
{
}
/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       sends outgoing message to appropriate port
 * \return      none
 * \retval      void
 * \param       param - pointer to the cresnet packet
 */
void CresnetSlaveEnqueueOutgoingPacket(UINT8 bStream, void *packet, UINT32 packet_size,
                                       UINT32 destination)
{
    //UINT32 status;
    //UINT32 timeout;
    void * CIP_target;

    if (IsDmFieldDebugIndexActive(DM_FIELD_CRESNET_TX_IDX))
		DmConsoleHexDumpPacket((UINT8*)packet, packet_size, "CnetTx:", 0);


    if ( destination == (UINT32)(NET_DEST_CUSTOM + CUSTOM_OFFSET) )
    {
        if ( m_pfCnetCustom )
        {
            // External processing for custom (DGE-2) to send the data to the PC
            (*m_pfCnetCustom)( packet );
        }
        return;
    }

    if((destination == NET_DEST_ALL) || (destination == NET_DEST_CRESNET))
    {
        if(pDMNetMapper)
        {
            pDMNetMapper->QueuePacketToController(bStream, packet, packet_size);
        }

    }
    if((destination == NET_DEST_ALL) || (destination == NET_DEST_CIP))
    {
        CIP_target = FindMasterByDeviceID(NET_DEST_CIP);
        if(CIP_target)
        {
            CipSlaveSendPacket((MASTER_LIST_ENTRY *)CIP_target,(UINT8*)packet, packet_size);
        }
    }
    else if(destination != NET_DEST_CRESNET)
    {
        /* check for destination device in IP table */
        CIP_target = FindMasterByDeviceID(destination);
        if(CIP_target)
        {
            // now also send it over the CIP
            CipSlaveSendPacket((MASTER_LIST_ENTRY *)CIP_target,(UINT8*)packet,packet_size);
        }
    }
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       send digital to outgoing queue
 * \return      none
 * \retval      void
 * \param       bStream - stream that's sending the join
 * \param       join - number of join to send
 * \param       on - true if join is set
 * \param       dest - which port to send it to
 *
 */
int CresnetSlaveSendDigital(UINT8 bStream, UINT32 join, BOOL on, UINT32 dest)
{
    UINT8 packet[16];
    UINT8 * p = packet;
    UINT8 slot;
    UINT8 * pLen1 = 0;
    UINT8 * pLen2 = 0;


    if(CresnetSlave.debug & 1)
    {
        DmConsolePrintf("CNET: tx digital(%d)=%u \r", join, on);
    }
    if(CresnetSlave.debug & 2)
        return 0;

    *p++ = CNET_ID_CTRLSYS;   /* dest ID */

    pLen1 = p;

    *p++ = 0;   /* packet length - fill in later */

    // check if this is a subslotted Join
    if( IsSubSlottedJoin(join) )
    {
        // need to encapsulate to convey slot and subslot info
        slot = SUBSLOT_FROM_JOIN_NUM(join);
        *p++ = CRESNET_ENCAPSULATED;
        *p++ = slot;
        pLen2 = p;
        *p++ = 0;  // len - fill in later

        // isolate the join number (remove subslot info)
        join = (UINT32)(join & SUBSLOT_JOIN_MASK);
    }
    else
       // now clear upper bits
       join = (UINT32)(join & SUBSLOT_JOIN_MASK);

    /* check for valid network join number */
    //if (join >= MAXBUT)
    //    return -1;

    *p++ = CNET_DIGITAL;   /* packet type */

    *p++ = (UINT16)join & 0x00FF;

    *p = ((UINT16)join >> 8) & 0x007F;
    if (!on)
    {
        *p |= 0x80;
    }
    p++;

    // fill in missing lengths as needed
    if(pLen1)
    {
        *pLen1 = p-(pLen1+1);
    }
    if(pLen2)
    {
        *pLen2 = p-(pLen2+1);
    }

    CresnetSlaveEnqueueOutgoingPacket(bStream, packet, p-packet, dest);

    return 0;
}

/**
 * \author      S.Novick
 * \date        10/25/2012
 * \brief       send repeat digital to outgoing queue
 * \return      none
 * \retval      void
 * \param       bStream - stream that's sending the join
 * \param       join - number of join to send
 * \param       on - true if join is set
 * \param       dest - which port to send it to
 */
int CresnetSlaveSendDigitalRepeat( UINT8 bStream, UINT32 join, BOOL on, UINT32 dest )
{
    UINT8 message[10];
    UINT8 *ptr = message;
    UINT8 subslot;

//    if (IsCrestnetSlaveQueueFull(bStream, dest))
//      return false;

    subslot = SUBSLOT_FROM_JOIN_NUM(join);
    join = JOIN_NUM_FROM_SLOTTED_JOIN(join);

    if (CresnetSlave.debug & 1)
    {
        if (subslot)
            DmConsolePrintf("CNET: tx digital %d.%d = %u \r", subslot, join, on);
        else
            DmConsolePrintf("CNET: tx digital %d = %u \r", join, on);
    }

    if ( !on )
    {
        join |= 0x8000;
    }

    *ptr++ = CNET_ID_CTRLSYS;
    if (subslot)
    {
        *ptr++ = 6;                       // lenght total
        *ptr++ = CRESNET_ENCAPSULATED;
        *ptr++ = subslot;
    }
    *ptr++ = 3;                         // length for type 27
    *ptr++ = CNET_DIGITAL_REPEAT;
    *ptr++ = join;
    *ptr++ = join >> 8;

    CresnetSlaveEnqueueOutgoingPacket(bStream, message, ptr - message, dest);
    return true;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       send analog to outgoing queue
 * \return      none
 * \retval      void
 * \param       bStream - stream that's sending the join
 * \param       join - number of join to send
 * \param       level - value of join
 * \param       dest - which port to send it to
 */
int CresnetSlaveSendAnalog(UINT8 bStream, UINT32 join, UINT16 level, UINT32 dest)
{
    UINT8 packet[16];
    UINT8 * p = packet;
    UINT8 * pLen1 = 0;
    UINT8 * pLen2 = 0;
    UINT8 slot;
    UINT16 channel;

    if(CresnetSlave.debug & 1)
    {
        DmConsolePrintf("CNET: tx analog(%d)=%u \r", join, level);
    }
    if(CresnetSlave.debug & 2)
        return 0;

    *p++ = CNET_ID_CTRLSYS;   /* destination ID */
    pLen1 = p;
    *p++ = 0;   /* packet length */

    // check if this is a subslotted Join
    if( IsSubSlottedJoin(join) )
    {
        // need to encapsulate to convey slot and subslot info
        slot = SUBSLOT_FROM_JOIN_NUM(join);
        *p++ = CRESNET_ENCAPSULATED;
        *p++ = slot;
        pLen2 = p;
        *p++ = 0;  // len - fill in later

        // clear upper bits
        channel = (UINT16)(join & SUBSLOT_JOIN_MASK);
    }
    else
        channel = (UINT16)(join & 0x0000ffff);

     /* use the longer packet type. It only supports the 2 Byte Channel, 2 Byte Data form*/
    *p++ = ANALOG_SYMMETRIC;   /* packet type */
    *p++ = HIUINT8(channel);
    *p++ = LOUINT8(channel);
    *p++ = HIUINT8(level);
    *p++ = LOUINT8(level);

    // fill in missing lengths as needed
    if(pLen1)
    {
        *pLen1 = p-(pLen1+1);
    }
    if(pLen2)
    {
        *pLen2 = p-(pLen2+1);
    }

    CresnetSlaveEnqueueOutgoingPacket(bStream, packet, p-packet, dest);

    return 0;
}

BOOL IsIso8859Disabled(void)
{
    return CresnetSlave.disableIso8859;
}

/**
 * \author      Pete McCormick
 *
 * \date        11/1/2005
 *
 * \return      int
 *
 * \retval      0: success
 * \retval      <0: failure
 *
 * \brief       Sends a Serial packet to the Control System
 *
 * \param       pMsg
 * \param       msgLen
 * \param       join
 * \param       dest
 *
 * \note
 *
 * \warning
 *
 */
int CresnetSlaveSendSerial(UINT8 bStream, UINT8 * pMsg, UINT32 msgLen, UINT32 join, UINT32 dest)
{
#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
    UINT8 bDMNetIdx;
#endif

    UINT8 * packet;
    UINT8 * p;
    UINT8 * pLen1 = 0;
    UINT8 * pLen2 = 0;
    UINT8 * pFlag = 0;
    UINT8 slot;
    UINT32 maxMsgLen;
    UINT32 lcount;
    UINT32 lTimeout;
    UINT8 packetHeaderSize = 0;
    int i,j;

    if(CresnetSlave.debug & 1)
    {
        int idx;
        DmConsolePrintf("tx serial %d length = %d \r", join, msgLen);
        for (idx = 0; idx<msgLen; idx++)
            DmConsolePrintf("\t0x%x\r", pMsg[idx]);
    }

    UINT16 packetSize = min( msgLen + SERIAL_HEADER_SIZE + ENCAP_OFFSET, MAX_CRESNET_MSG_OUT );

    //no sense in going further if there aren't enough buffers for this function and the queue
    if( !MemMgr || MemMgr->BuffersAvailable( packetSize ) < 2 )
        return -1;

    // get temporary buffer
    packet = GetCresnetBuffer( packetSize );
    p = packet;

    if( !p )
        return -1;

    *p++ = CNET_ID_CTRLSYS;   /* destination ID */
    pLen1 = p;
    *p++ = 0;   /* packet length - fill in later */
    packetHeaderSize = 2; // 2 for ID, length

    // check if this is a subslotted Join
    if( IsSubSlottedJoin(join) )
    {
        // need to encapsulate to convey slot and subslot info
        slot = SUBSLOT_FROM_JOIN_NUM(join);

#ifdef DM_MIDPOINT_DEVICE_CONTROLLER_SUPPORT
	    
			if(GetSlotFromDMNetIdx(&bDMNetIdx, slot, CtrlrConfig.bMaxOutputSlotNum, 
				   CtrlrConfig.bMaxInputSlotNum)==0)
			    {
				slot = bDMNetIdx - 1;
/*				lJoin =  SET_SUBSLOT_BITS(JOIN_NUM_FROM_SLOTTED_JOIN(lJoin), 
							  bPacketSlot); 	
	*/		    }
#endif                        



        *p++ = CRESNET_ENCAPSULATED;
        *p++ = slot;
        pLen2 = p;
        *p++ = 0;  // len - fill in later
        packetHeaderSize += 3; // 3 for CRESNET_ENCAPSULATED, slot, length

        // isolate the join number (remove subslot info)
        join = (UINT32)(join & SUBSLOT_JOIN_MASK);
    }
    else
       // now clear upper bits
       join &= 0x0000ffff;


    *p++ = CRESNET_SERIAL;   /* packet type */
    *p++ = (UINT8)((join & 0xFF00)>>8 );   /* serial channel hi*/
    *p++ = (UINT8)(join & 0xFF); /* serial channel low */

    pFlag = p;
    *p++ = 0; /* packet flag - fill in later */

    packetHeaderSize += 4; // 4 for type, ch hi, ch low, flag

    maxMsgLen = MAX_CRESNET_MSG_OUT - packetHeaderSize;


    if(msgLen<maxMsgLen)
        lcount = msgLen;
    else
        lcount = maxMsgLen;

    // if there's no data, just send header
    if (msgLen == 0)
    {
          // fill in missing lengths as needed
          if(pLen1)
          {
              *pLen1 = p-(pLen1+1);
          }
          if(pLen2)
          {
              *pLen2 = p-(pLen2+1);
          }
          *pFlag = START_AND_END;
          CresnetSlaveEnqueueOutgoingPacket(bStream, packet, p-packet, dest);
    }
    else // send data in one or more packets
    {
      for(i=0; i<msgLen; i+=lcount)
      {
          // Reset pointer to the data section
          p = packet + packetHeaderSize;

          for(j=0; (j<lcount) &&((i+j)<msgLen); j++)
          {
              *p++ = *pMsg++;
          }

          // fill in packet flag
          if(i==0)   // first message packet
          {
              if(j<msgLen)
                  *pFlag = START_NOT_END;
              else
                  *pFlag = START_AND_END;
          }
          else
          {
              if((i+j)<msgLen)
                  *pFlag = APPEND_NOT_END;
              else
                  *pFlag = APPEND_AND_END;
          }


          // fill in missing lengths as needed
          if(pLen1)
          {
              *pLen1 = p-(pLen1+1);
          }
          if(pLen2)
          {
              *pLen2 = p-(pLen2+1);
          }

         // Always check, to make sure the last packet sent is out, this is address senario's where the applicaiton sends back to back packets
         // and the driver has a single memory buffer.

         // Reset timeout counter
         lTimeout = CRESNET_QUEUE_TIMEOUT_10MS;

         // Wait for the queue to be ready or a timeout
         while(IsCrestnetSlaveQueueFull(bStream, dest) && lTimeout > 0)
         {
            HwDelayMsec(CRESNET_QUEUE_POLL_RATE);
            lTimeout--;
         }

         // Check for a timeout
         if (lTimeout == 0)
         {
            // Error log
            DmSystemError(DM_ERROR_LEVEL_WARNING, DM_ERROR_SUBSYS_CNET, ERR_CNET_DROP_PKT_SENDING, 1);
         }

         CresnetSlaveEnqueueOutgoingPacket(bStream, packet, p-packet, dest);
      }
    }
    // free up temporary buffer
    ReleaseCresnetBuffer(packet);

    return 0;
}

void CresnetDisableIso8859(BOOL value)
{
    CresnetSlave.disableIso8859 = value;
}
/**
 * \author    Larry Salant
 * \brief     enables / disable Cresnet debug mode (send cmds to console)
 * \date      4/16/2008
 * \param     bEnable - 0 disables the debugging;  non zero enables
 * \return    void
 * \retval    none
**/
void CresnetSlaveDebugEnable(UINT8 bEnable)
{
  CresnetSlave.debug = bEnable;
}
/**
 * \author    Larry Salant
 * \brief     in response to a capability request, send capability packet
 * <id> <cnt> <31> <Type> <Flags> <Transaction#> <Configuration string definition>
 *      <Type> is:   XX:    reserved
 *      <Transaction#> is a unique byte identifier & must match transaction number of capability request packet.
 *      <Flags> is:   00:   last packet in response
 *                    01:   more capability packets to follow
 * \date      4/1/2007
 * \param     bRequest - request packet  <type> <Flags> <Transaction #>
 *                  type is reserved, Flags are for symmetry with response packet
 *                  <Transaction#> is a unique byte id.
 * \return    void
 * \retval    none
**/
int CresnetSlaveSendCapabilities(UINT8 bStream, COMMAND_RESPONSE *bRequest)
{
    UINT8 * packet;
    UINT8 * p;
    UINT16 sLen;
    UINT8 * pLen1 = 0;
    char const *pCapabilities;

    // get pointer to capabilites for this stream
    pCapabilities = GetCapabilityString(bStream);
    sLen = strlen(pCapabilities);

    if (sLen > MAX_CAPABILITY_LENGTH)
        sLen = MAX_CAPABILITY_LENGTH;

    //no sense in going further if two buffers aren't free
    if( !MemMgr || MemMgr->BuffersAvailable( sLen + CAPABILITY_HEADER_SIZE ) < 2 )
        return -1;

    packet = GetCresnetBuffer( sLen + CAPABILITY_HEADER_SIZE );
    p = packet;

    if( !p )
        return -1;

    // build the response packet
    *p++ = CNET_ID_CTRLSYS;   /* destination ID */
    pLen1 = p;  //this is the length byte, store address for future use
    *p++ = 0;   // packet length - fill in later
    *p++ = CAPABILITY_RESPONSE;
    *p++ = CAPABILITY_BASIC_TYPE;
    // if this is the last packet
    if (bStream == (CresnetSlave.NumStreams-1))
      *p++ = CAPABILITY_LAST_PACKET;
    else
      *p++ = CAPABILITY_MORE_PACKETS;

    // add the transaction number from the request
    *p++ = bRequest->Transaction;

    memcpy((char *)p,pCapabilities,sLen);
    p += sLen;

    // fill in length
    if(pLen1)
      *pLen1 = p-(pLen1+1);

    // send response
    CresnetSlaveEnqueueOutgoingPacket(bStream, packet, p-packet, bRequest->Source);

    // free up temporary buffer
    ReleaseCresnetBuffer(packet);

    return 0;
}
/**
 * \author    Adolfo Velasco
 * \brief     in response to a exended capability request, send extended capability packet
 * <id> <cnt> <44> <Type> <Flags> <Transaction#> <Configuration string definition>
 *      <Type> is:   XX:    reserved
 *      <Transaction#> is a unique byte identifier & must match transaction number of capability request packet.
 *      <Flags> is:   00:   last packet in response
 *                    01:   more capability packets to follow
 * \date      11/12/2012
 * \param     bRequest - request packet  <type> <Flags> <Transaction #>
 *                  type is reserved, Flags are for symmetry with response packet
 *                  <Transaction#> is a unique byte id.
 * \return    void
 * \retval    none
**/
int CresnetSlaveSendExtendedCapabilities(UINT8 bStream, COMMAND_RESPONSE *bRequest)
{
    UINT8 * packet;
    UINT8 * p;
    UINT16 sLen;
    UINT8 * pLen1 = 0;
    char const *pCapabilities;

    // get pointer to capabilites for this stream
    pCapabilities = GetExtendedCapabilityString(bStream);
    if ( !pCapabilities )
    {
        sLen = 0;
    }
    else
    {
        sLen = strlen(pCapabilities);
    }

    if (sLen > MAX_CAPABILITY_LENGTH)
        sLen = MAX_CAPABILITY_LENGTH;

    //no sense in going further if two buffers aren't free
    if( !MemMgr || MemMgr->BuffersAvailable( sLen + CAPABILITY_HEADER_SIZE ) < 2 )
        return -1;

    packet = GetCresnetBuffer( sLen + CAPABILITY_HEADER_SIZE );
    p = packet;

    if( !p )
        return -1;

    // build the response packet
    *p++ = CNET_ID_CTRLSYS;   /* destination ID */
    pLen1 = p;  //this is the length byte, store address for future use
    *p++ = 0;   // packet length - fill in later
    *p++ = EXTENDED_CAPABILITY_RESPONSE;
    *p++ = CAPABILITY_BASIC_TYPE;
    // if this is the last packet
    if (bStream == (CresnetSlave.NumStreams-1))
      *p++ = CAPABILITY_LAST_PACKET;
    else
      *p++ = CAPABILITY_MORE_PACKETS;

    // add the transaction number from the request
    *p++ = bRequest->Transaction;


    if ( pCapabilities && sLen )
    {
        memcpy((char *)p,pCapabilities,sLen);
    }
    p += sLen;

    // fill in length
    if(pLen1)
      *pLen1 = p-(pLen1+1);

    // send response
    CresnetSlaveEnqueueOutgoingPacket(bStream, packet, p-packet, bRequest->Source);

    // free up temporary buffer
    ReleaseCresnetBuffer(packet);

    return 0;
}
/**
 * \author    Larry Salant
 * \brief     if there is room in the output queue and there are joins that haven't been sent out,
 *            builds the packets and sends them.
 * \date      4/1/2007
 * \param     none
 * \return    void
 * \retval    none
**/

static UINT8 lastStreamPolled = 0;  // try to balance streams, if more then one

void CresnetSlaveSendPending(void)
{
    UINT8 bCount;
    UINT8 bMsgSent = 0;
    UINT8 bStream;

    // for each stream
    for (bStream = 0; bStream < CresnetSlave.NumStreams; bStream++)
    {
        //Check for the command sequence request ack
        if ( GetState(bStream) )
        {
            GetState(bStream)->CheckCommandRequestAckTimer();
        }

        // check if there are any joins that have changed that have not been sent
        // stop if a packets are sent to give it a chance to send
        for (bCount = 0; bCount < MAX_MESSAGE_TYPES && !bMsgSent; ++bCount)
        {

            if(pDMNetMapper)
            {
                // give priority to joins coming from master polling us and dircted to the switch
                bMsgSent = pDMNetMapper->SendPendingToController(lastStreamPolled);
            }
                        
            // rotate thru the types so they all get a chance to be sent
            switch (NextType)
            {
                case COMMAND_CHECK:
                    // if there is a digital that has changed, send the message
                    bMsgSent = GetState(lastStreamPolled)->CresnetNextCommandPending();
                    break;
                case DIGITAL_CHECK:
                    // if there is a digital that has changed, send the message
                    bMsgSent = GetState(lastStreamPolled)->CresnetNextDigitalPending();
                    break;
                case ANALOG_CHECK:
                    // if there is a analog that has changed, send the message
                    bMsgSent = GetState(lastStreamPolled)->CresnetNextAnalogPending();
                    break;
                case SERIAL_CHECK:
                    // if there is a serial that has changed, send the message
                    bMsgSent = GetState(lastStreamPolled)->CresnetNextSerialPending();
                    break;
            }

            // increment to next message type
            NextType++;
            if (NextType >= MAX_MESSAGE_TYPES)
                NextType = 0;
        }

        lastStreamPolled = (lastStreamPolled + 1)%CresnetSlave.NumStreams;
    }
}

/**
 * \author      Larry Salant
 * \date        5/6/2008
 * \brief       check if the queue for this stream is filled
 * \param       void
 * \return      UINT8
 * \retval      0 if the queue has room
 */
UINT8 IsCrestnetSlaveQueueFull(UINT8 bStream, UINT8 bDestination)
{
    // if the destination is cresnet,
    if (bDestination == NET_DEST_CRESNET)
    {
        if(pDMNetMapper)
        {
            // check the queue for the Tx connected to the Controller
            return pDMNetMapper->IsControllerAvailable(bStream);
        }
		return false;

    }
    if (bDestination == (UINT8)(NET_DEST_CUSTOM + CUSTOM_OFFSET) )
    {
        return false;
    }
    else // CIP - check if it's on line
        return IsCIPQueueFull(bDestination);
}
/**
 * \author    Larry Salant
 * \brief     gets the lock for the CresentSlave Output Queue
 * \date      4/1/2007
 * \param     none
 * \return    UINT8*
 * \retval    pointer to buffer
**/
void GetCresnetSlaveLock( void)
{
  OsLock(CresnetSlaveLockId);
}
/**
 * \author    Larry Salant
 * \brief     releases the lock for the CresentSlave Output Queue
 * \date      4/1/2007
 * \param     none
 * \return    UINT8*
 * \retval    pointer to buffer
**/
void ReleaseCresnetSlaveLock( void)
{
  OsUnlock(CresnetSlaveLockId);
}

/**
 * \author    Larry Salant
 * \brief     reserves the temporary cresnet buffer
 * \date      4/1/2007
 * \param     none
 * \return    UINT8*
 * \retval    pointer to buffer
**/
UINT8* GetCresnetBuffer( UINT16 size )
{
  UINT8 *bCresnetBuffer = 0;

  if (MemMgr)
    bCresnetBuffer = MemMgr->GetBlock( size );

  return bCresnetBuffer;
}
/**
 * \author    Larry Salant
 * \brief     reserves the temporary cresnet buffer
 * \date      4/1/2007
 * \param     pointer to buffer
 * \return    void
 * \retval    none
**/
void ReleaseCresnetBuffer(UINT8 *buffer)
{
    // free the buffer used for this command
    if (MemMgr)
      MemMgr->FreeBlock(buffer);
}

//^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
// Type 29h: Shaft Encoder
//
// <id> <07> <29> <join-hi> <join-lo> <time-hi> <time-lo> <delta-hi> <delta-lo>
//
// <delta> is a signed displacement of how far the knob was turned (if at all) since the last poll.
// <time> is the time (in units of 10ms) since the last poll, which with the delta, would tell how
// fast the knob was being turned.  The control system could then make the change to the analog
// signal that the knob was driving proportional to the knob speed.
//
// The packet processing just squared the delta of -32768 to +32767, multiplied by 64, then divided
// by the time to get the delta, i.e.  dx = K * delta^2 / time.  K = 64.  dx is then applied to the
// analog value (remembering the sign of the delta).
//
BOOL CresnetSlaveSendType29( UINT8 bStream, UINT32 join, UINT16 time, UINT16 delta, UINT32 dest )
{
    UINT8 message[16];
    UINT8 Len = 0;

    if(IsCrestnetSlaveQueueFull(bStream, dest))
      return false;

    if(CresnetSlave.debug & 1)
    {
        DmConsolePrintf("CNET: tx Type29 join=%d, time=%d, delta=%d \r", join, time, delta);
    }
    if(CresnetSlave.debug & 2)
        return 0;

    // check if this is a subslotted Join
    if( IsSubSlottedJoin(join) )
    {
        message[0] = CNET_ID_CTRLSYS;   /* destination ID */
        message[1] = 10;             // length encapsulated
        message[2] = CRESNET_ENCAPSULATED;           // encapsulation
        message[3] = SUBSLOT_FROM_JOIN_NUM(join);//slot
        message[4] = 7;              // length original
        message[5] = SHAFT_ENCODER_PACKET;           // packet type, finally
        // isolate the join number (remove subslot info)
        join = (UINT32)(join & SUBSLOT_JOIN_MASK);

        message[6] = HIUINT8( join );
        message[7] = LOUINT8( join );
        message[8] = HIUINT8( time );
        message[9] = LOUINT8( time );
        message[10] = HIUINT8( delta );
        message[11] = LOUINT8( delta );
        Len = 12;

    }
    else            // slot == 0, not encapsulated
    {
        message[0] = CNET_ID_CTRLSYS;   /* destination ID */
        message[1] = 7;
        message[2] = SHAFT_ENCODER_PACKET;
        message[3] = HIUINT8( join );
        message[4] = LOUINT8( join );
        message[5] = HIUINT8( time );
        message[6] = LOUINT8( time );
        message[7] = HIUINT8( delta );
        message[8] = LOUINT8( delta );
        Len = 9;
    }
    CresnetSlaveEnqueueOutgoingPacket(bStream, message, Len, dest);
    return true;
}

//^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
// Type 2Bh: Temperature Reporting
//
// <id> <05> <2B> <ch-hi> <ch-lo> <temp-hi> <temp-lo>
//
// The <temp.hi> <temp.lo> is reported in degree Celsius multiplied by 10.
// Channels start at 0, but are dependent on the device in question.
//
BOOL CresnetSlaveSendType2B( UINT8 bStream, UINT16 ch, UINT16 temp, UINT32 dest )
{
    UINT8 message[7];
    UINT8 Len = 0;

    if(IsCrestnetSlaveQueueFull(bStream, dest))
      return false;

    if(CresnetSlave.debug & 1)
    {
        DmConsolePrintf("CNET: tx Type2B temp=%d ch=%d\r\n", ch, temp);
    }

    message[0] = CNET_ID_CTRLSYS;   /* destination ID */
    message[1] = 5;
    message[2] = TEMPERATURE;
    message[3] = HIUINT8( ch );
    message[4] = LOUINT8( ch );
    message[5] = HIUINT8( temp );
    message[6] = LOUINT8( temp );
    Len = 7;
    CresnetSlaveEnqueueOutgoingPacket(bStream, message, Len, dest);
    return true;
}

/*------------------------------------------------------------------------------------------------*
For fixed format Cresnet II RF, the bit format is:
Type 09h: 1-Way RF Receiver

<id> <cnt> <09> <data2> <data1> <data0>

For 1-Way receivers reporting back to the Cresnet Master (ID 02), a typical packet is:
<02> <04> <09> <data2> <data1> <data0>

In response to a poll, data field is zero if no signal present, else it is framed 24b data.

For fixed format Cresnet II RF, the bit format is:

 011110xx x0xxx0xx x0xxx0xx
       ii i iii ii k kkk kk
       76 5 432 10 5 432 10

Where iiiiiiii is the 8-bit RF ID code, and kkkkkk is a 6 bit # (join 1-64) of the button #.
*-------------------------------------------------------------------------------------------------*/
/**
 * \author    Dennis Hromin
 * \brief     Builds and Sends out Cresnet Return Packet of type 0x09 RF1Way Data
 * \date      12/30/2008
 * \param     bStream   Bit Stream to report back on
 * \param     join  Join number to report back on
 * \param     data0,1,2 Three data bytes for RF ID/packet or RC5 code
 * \param     dest  Destination where to send pkt out (CNET or CIP Ethernet)
 * \return    void
 * \retval    none
**/
BOOL CresnetSlaveSendType09( UINT8 bStream, UINT32 join, UINT8 data0, UINT8 data1, UINT8 data2, UINT32 dest )
{
    UINT8 message[16];
    UINT8 Len = 0;

    if(IsCrestnetSlaveQueueFull(bStream, dest))
      return false;

    if(CresnetSlave.debug & 1)
    {
        DmConsolePrintf("CNET: tx Type09 Data = %x%x%x \r", data0, data1, data2);
    }
    if(CresnetSlave.debug & 2)
        return 0;

    // check if this is a subslotted Join
    if( IsSubSlottedJoin(join) )
    {
        message[0] = CNET_ID_CTRLSYS;   /* destination ID */
        message[1] = 07;             // length encapsulated
        message[2] = CRESNET_ENCAPSULATED;           // encapsulation
        message[3] = SUBSLOT_FROM_JOIN_NUM(join);//slot
        message[4] = 04;              // length original
        message[5] = RF_DAT;           // packet type, finally
        message[6] = data0;
        message[7] = data1;
        message[8] = data2;
        Len = 9;

    }
    else // slot == 0, not encapsulated
    {
        message[0] = CNET_ID_CTRLSYS;   /* destination ID */
        message[1] = 04;         // length original
        message[2] = RF_DAT;    // packet type, finally
        message[3] = data0;
        message[4] = data1;
        message[5] = data2;
        Len = 6;
    }
    CresnetSlaveEnqueueOutgoingPacket(bStream, message, Len, dest);
    return true;
}

/**
 * \author    Adolfo  Velasco
 * \brief     Processes the transfer start
 * \date      03/03/2009
 * \param     none
 * \return    void
 * \retval    none
**/
void CresnetProcessTransferStart( void )
{
    UINT8 i;

    //Verify stream mapper class since we will be using it
    if ( !g_pStreamMapper )
        return;

    //Disable all the streams on this card
    for ( i = 0; i < g_pStreamMapper->GetStreamCount(); i++ )
    {
        g_pStreamMapper->DisableStream(i);
    }
}

/**
 * \author    Adolfo  Velasco, Rob Carter
 * \brief     Processes the transfer end
 * \date      03/03/2009
 * \param     none
 * \return    void
 * \retval    none
**/
void CresnetProcessTransferEnd( UINT8 bTransferAborted )
{
    //If transfer was not aborted then reboot
    if ( !bTransferAborted )
    {
        DmRebootNotFatal();
    }

    //This code enables the streams
    /*
    UINT8 i;

    //Verify stream mapper class since we will be using it
    if ( !g_pStreamMapper )
        return;

    //Enable all the streams on this card
    for ( i = 0; i < g_pStreamMapper->GetStreamCount(); i++ )
    {
        g_pStreamMapper->EnableStream(i);
    }
    */
}

#ifdef STR91X_IAR
/**
* \author      Dariusz Rymsza
* \date        3/18/2009
* \return      void
* \retval      void
* \brief       Write first block of bootloader upgrade
* \param       none
*
*/
void WriteBootloaderVectorBlockIfNeeded(void)
{
  bool bWriteErrors =false;

  if (gbBufferingVectorTableSRecords && gpBootloaderVectorArea[0] && gpBootloaderVectorArea[1])
  {
    // writing of the bootloader vector area has to be an uninterruptable operation
    // if we get an interrupt during this process, it will get routed to
    // an undefined location
    __disable_interrupt();

    gbWritingBootloaderVectorArea = TRUE; // flag to everybody else to NOT enable interrupts

    EraseBootLoader();  // erase the bootloader

    // start writing at address 0, beginning of flash, write both buffers
    int startAddr = 0;
    for (int i = 0; i < 2; i++)
    {
      if(FlashWriteBlock(startAddr, gpBootloaderVectorArea[i], gBootloaderVectorAreaByteCnt[i]) != 0)
      {
        if (gpCnetHelper) // can't abort here, interrupts are disabled
        {
          gpCnetHelper->xferWriteErrCnt++;
          bWriteErrors = true;
        }
      }
      startAddr += gBootloaderVectorAreaByteCnt[i];
    }

    // done writing
    gbWritingBootloaderVectorArea = FALSE;  // we are free to enable interrupts
    gbBufferingVectorTableSRecords = false;
    gpCnetHelper = NULL;

    __enable_interrupt();

    // abort, do it here when interrupts are enabled again
    if (bWriteErrors)
    {
        EnableNVLWrites();
        CresnetDataXferAbort();
        CresnetFlashLoaderSendAbortXfer();
    }
  }
}

/**
* \author      Dariusz Rymsza
* \date        3/18/2008
* \return      void
* \retval      void
* \brief       Handle writing of bootloader S records to flash
* \param       addr
* \param       pData
* \param       byteCnt
* \note
* \note        when upgrading bootloader we have to erase the flash and write the vectors
*              and their service code in one uninterruptable operation
*              buffer the first block send here, than write it when we receive block end
*              S record data is kept in a linked list structure, whith each node holding
*              data for one S record
*/
void WriteBootloaderFlashRecord(UINT32 addr, UINT8 * pData, UINT8 byteCnt, CNETHELPER *pCnetHelper)
{
  gpCnetHelper = pCnetHelper;

  if (gbBufferingVectorTableSRecords)   // we are buffering
  {
    if (gpBootloaderVectorArea[0] == NULL)  // check if have already allocated buffers, it's enought to check the first one
    {
      gBootloaderVectorAreaByteCnt[0] = 0;   // keep count of bytes buffered so far
      gBootloaderVectorAreaByteCnt[1] = 0;
      // we need to save around 512 bytes, need two XL buffers to do that
      if (!MemMgr || (gpBootloaderVectorArea[0] = MemMgr->GetBlock(EXTRA_LARGE_BLOCK_LEN)) == 0 || (gpBootloaderVectorArea[1] = MemMgr->GetBlock(EXTRA_LARGE_BLOCK_LEN)) == 0)
      {
        pCnetHelper->xferWriteErrCnt++;

        if (MemMgr && gpBootloaderVectorArea[0])
          MemMgr->FreeBlock(gpBootloaderVectorArea[0]);

        CresnetDataXferAbort();
        CresnetFlashLoaderSendAbortXfer();
        EnableNVLWrites();
      }
    }

    // make sure that blocks are consecutive, we don't want to leave any "holes" in the flash
    if (addr != pCnetHelper->writeAddr)
    {
      pCnetHelper->xferWriteErrCnt++;
      CresnetDataXferAbort();
      CresnetFlashLoaderSendAbortXfer();
      EnableNVLWrites();
      return;
    }
    // figure out which buffer we are going to write to
    int areaIdx;
    if ((addr + byteCnt) > EXTRA_LARGE_BLOCK_LEN)
      areaIdx = 1;
    else
      areaIdx = 0;

    // buffer packet data
    // if placing in 2nd buffer, subtract from current address bytes buffered so far, this gives us zero based index in 2nd buffer
    for (int i = 0; i < byteCnt; i++)
      gpBootloaderVectorArea[areaIdx][(addr - areaIdx*gBootloaderVectorAreaByteCnt[0]) + i] = pData[i];

    gBootloaderVectorAreaByteCnt[areaIdx] += byteCnt;   // keep count of bytes buffered so far
  }
  else  // no longer buffering, just write it
  {
    if(FlashWriteBlock(addr, pData, byteCnt) != 0)
      pCnetHelper->xferWriteErrCnt++;
  }
}

/**
* \author      Dariusz Rymsza
* \date        3/18/2009
* \return      void
* \retval      void
* \brief       Clean up temporary buffers after bootloader upgrade
* \param       none
*
*/
void ClearBootloaderSRecordData(void)
{
  gbBufferingVectorTableSRecords = false;   // don't buffer S records anymore
  gbWritingBootloaderVectorArea = FALSE;    // we are free to enable interrupts
  gpCnetHelper = NULL;

  for (int i = 0; i < 2; i++)
  {
    if (gpBootloaderVectorArea[i])
    {
      MemMgr->FreeBlock(gpBootloaderVectorArea[i]);
      gpBootloaderVectorArea[i] = NULL;
    }
  }
}

#endif // STR91X_IAR

void SetCnetCustomFunctionPointer( CNET_CUSTOM pfCnetCustom )
{
    m_pfCnetCustom = pfCnetCustom;
}

